---
toc_title: H3 Indexes
---

# Functions for Working with H3 Indexes {#h3index}

[H3](https://eng.uber.com/h3/) is a geographical indexing system where Earth’s surface divided into a grid of even hexagonal cells. This system is hierarchical, i. e. each hexagon on the top level ("parent") can be splitted into seven even but smaller ones ("children"), and so on.

The level of the hierarchy is called `resolution` and can receive a value from `0` till `15`, where `0` is the `base` level with the largest and coarsest cells.

A latitude and longitude pair can be transformed to a 64-bit H3 index, identifying a grid cell.

The H3 index is used primarily for bucketing locations and other geospatial manipulations.

The full description of the H3 system is available at [the Uber Engeneering site](https://eng.uber.com/h3/).

## h3IsValid {#h3isvalid}

Verifies whether the number is a valid [H3](#h3index) index.

**Syntax**

``` sql
h3IsValid(h3index)
```

**Parameter**

-   `h3index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   1 — The number is a valid H3 index.
-   0 — The number is not a valid H3 index.

Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT h3IsValid(630814730351855103) AS h3IsValid;
```

Result:

``` text
┌─h3IsValid─┐
│         1 │
└───────────┘
```

## h3GetResolution {#h3getresolution}

Defines the resolution of the given [H3](#h3index) index.

**Syntax**

``` sql
h3GetResolution(h3index)
```

**Parameter**

-   `h3index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   Index resolution. Range: `[0, 15]`.
-   If the index is not valid, the function returns a random value. Use [h3IsValid](#h3isvalid) to verify the index.

Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT h3GetResolution(639821929606596015) AS resolution;
```

Result:

``` text
┌─resolution─┐
│         14 │
└────────────┘
```

## h3EdgeAngle {#h3edgeangle}

Calculates the average length of the [H3](#h3index) hexagon edge in grades.

**Syntax**

``` sql
h3EdgeAngle(resolution)
```

**Parameter**

-   `resolution` — Index resolution. Type: [UInt8](../../../sql-reference/data-types/int-uint.md). Range: `[0, 15]`.

**Returned values**

-   The average length of the [H3](#h3index) hexagon edge in grades. Type: [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT h3EdgeAngle(10) AS edgeAngle;
```

Result:

``` text
┌───────h3EdgeAngle(10)─┐
│ 0.0005927224846720883 │
└───────────────────────┘
```

## h3EdgeLengthM {#h3edgelengthm}

Calculates the average length of the [H3](#h3index) hexagon edge in meters.

**Syntax**

``` sql
h3EdgeLengthM(resolution)
```

**Parameter**

-   `resolution` — Index resolution. Type: [UInt8](../../../sql-reference/data-types/int-uint.md). Range: `[0, 15]`.

**Returned values**

-   The average length of the [H3](#h3index) hexagon edge in meters. Type: [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT h3EdgeLengthM(15) AS edgeLengthM;
```

Result:

``` text
┌─edgeLengthM─┐
│ 0.509713273 │
└─────────────┘
```

## h3EdgeLengthKm {#h3edgelengthkm}

Calculates the average length of the [H3](#h3index) hexagon edge in kilometers.

**Syntax**

``` sql
h3EdgeLengthKm(resolution)
```

**Parameter**

-   `resolution` — Index resolution. Type: [UInt8](../../../sql-reference/data-types/int-uint.md). Range: `[0, 15]`.

**Returned values**

-   The average length of the [H3](#h3index) hexagon edge in kilometers. Type: [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT h3EdgeLengthKm(15) AS edgeLengthKm;
```

Result:

``` text
┌─edgeLengthKm─┐
│  0.000509713 │
└──────────────┘
```

## geoToH3 {#geotoh3}

Returns [H3](#h3index) point index `(lon, lat)` with specified resolution.

**Syntax**

``` sql
geoToH3(lon, lat, resolution)
```

**Arguments**

-   `lon` — Longitude. Type: [Float64](../../../sql-reference/data-types/float.md).
-   `lat` — Latitude. Type: [Float64](../../../sql-reference/data-types/float.md).
-   `resolution` — Index resolution. Range: `[0, 15]`. Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   Hexagon index number.
-   0 in case of error.

Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT geoToH3(37.79506683, 55.71290588, 15) AS h3Index;
```

Result:

``` text
┌────────────h3Index─┐
│ 644325524701193974 │
└────────────────────┘
```

## h3ToGeo {#h3togeo}

Returns the centroid longitude and latitude corresponding to the provided [H3](#h3index) index.

**Syntax**

``` sql
h3ToGeo(h3Index)
```

**Arguments**

-   `h3Index` — H3 Index. [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   A tuple consisting of two values: `tuple(lon,lat)`. `lon` — Longitude. [Float64](../../../sql-reference/data-types/float.md). `lat` — Latitude. [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT h3ToGeo(644325524701193974) AS coordinates;
```

Result:

``` text
┌─coordinates───────────────────────────┐
│ (37.79506616830252,55.71290243145668) │
└───────────────────────────────────────┘
```

## h3ToGeoBoundary {#h3togeoboundary}

Returns array of pairs `(lon, lat)`, which corresponds to the boundary of the provided H3 index.

**Syntax**

``` sql
h3ToGeoBoundary(h3Index)
```

**Arguments**

-   `h3Index` — H3 Index. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-    Array of pairs '(lon, lat)'.
Type: [Array](../../../sql-reference/data-types/array.md)([Float64](../../../sql-reference/data-types/float.md), [Float64](../../../sql-reference/data-types/float.md)).


**Example**

Query:

``` sql
SELECT h3ToGeoBoundary(644325524701193974) AS coordinates;
```

Result:

``` text
┌─h3ToGeoBoundary(599686042433355775)────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
│ [(37.2713558667319,-121.91508032705622),(37.353926450852256,-121.8622232890249),(37.42834118609435,-121.92354999630156),(37.42012867767779,-122.03773496427027),(37.33755608435299,-122.090428929044),(37.26319797461824,-122.02910130919001)] │
└────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
```

## h3kRing {#h3kring}

 Lists all the [H3](#h3index) hexagons in the raduis of `k` from the given hexagon in random order.

**Syntax**

``` sql
h3kRing(h3index, k)
```

**Arguments**

-   `h3index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `k` — Radius. Type: [integer](../../../sql-reference/data-types/int-uint.md)

**Returned values**

-   Array of H3 indexes.

Type: [Array](../../../sql-reference/data-types/array.md)([UInt64](../../../sql-reference/data-types/int-uint.md)).

**Example**

Query:

``` sql
SELECT arrayJoin(h3kRing(644325529233966508, 1)) AS h3index;
```

Result:

``` text
┌────────────h3index─┐
│ 644325529233966508 │
│ 644325529233966497 │
│ 644325529233966510 │
│ 644325529233966504 │
│ 644325529233966509 │
│ 644325529233966355 │
│ 644325529233966354 │
└────────────────────┘
```

## h3GetBaseCell {#h3getbasecell}

Returns the base cell number of the [H3](#h3index) index.

**Syntax**

``` sql
h3GetBaseCell(index)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-   Hexagon base cell number.

Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT h3GetBaseCell(612916788725809151) AS basecell;
```

Result:

``` text
┌─basecell─┐
│       12 │
└──────────┘
```

## h3HexAreaM2 {#h3hexaream2}

Returns average hexagon area in square meters at the given resolution.

**Syntax**

``` sql
h3HexAreaM2(resolution)
```

**Parameter**

-   `resolution` — Index resolution. Range: `[0, 15]`. Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-   Area in square meters.

Type: [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT h3HexAreaM2(13) AS area;
```

Result:

``` text
┌─area─┐
│ 43.9 │
└──────┘
```

## h3HexAreaKm2 {#h3hexareakm2}

Returns average hexagon area in square kilometers at the given resolution.

**Syntax**

``` sql
h3HexAreaKm2(resolution)
```

**Parameter**

-   `resolution` — Index resolution. Range: `[0, 15]`. Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-   Area in square kilometers.

Type: [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT h3HexAreaKm2(13) AS area;
```

Result:

``` text
┌──────area─┐
│ 0.0000439 │
└───────────┘
```

## h3IndexesAreNeighbors {#h3indexesareneighbors}

Returns whether or not the provided [H3](#h3index) indexes are neighbors.

**Syntax**

``` sql
h3IndexesAreNeighbors(index1, index2)
```

**Arguments**

-   `index1` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `index2` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-   `1` — Indexes are neighbours.
-   `0` — Indexes are not neighbours.

Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT h3IndexesAreNeighbors(617420388351344639, 617420388352655359) AS n;
```

Result:

``` text
┌─n─┐
│ 1 │
└───┘
```

## h3ToChildren {#h3tochildren}

Returns an array of child indexes for the given [H3](#h3index) index.

**Syntax**

``` sql
h3ToChildren(index, resolution)
```

**Arguments**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `resolution` — Index resolution. Range: `[0, 15]`. Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   Array of the child H3-indexes.

Type: [Array](../../../sql-reference/data-types/array.md)([UInt64](../../../sql-reference/data-types/int-uint.md)).

**Example**

Query:

``` sql
SELECT h3ToChildren(599405990164561919, 6) AS children;
```

Result:

``` text
┌─children───────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
│ [603909588852408319,603909588986626047,603909589120843775,603909589255061503,603909589389279231,603909589523496959,603909589657714687] │
└────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
```

## h3ToParent {#h3toparent}

Returns the parent (coarser) index containing the given [H3](#h3index) index.

**Syntax**

``` sql
h3ToParent(index, resolution)
```

**Arguments**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `resolution` — Index resolution. Range: `[0, 15]`. Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-   Parent H3 index.

Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT h3ToParent(599405990164561919, 3) AS parent;
```

Result:

``` text
┌─────────────parent─┐
│ 590398848891879423 │
└────────────────────┘
```

## h3ToString {#h3tostring}

Converts the `H3Index` representation of the index to the string representation.

``` sql
h3ToString(index)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-   String representation of the H3 index.

Type: [String](../../../sql-reference/data-types/string.md).

**Example**

Query:

``` sql
SELECT h3ToString(617420388352917503) AS h3_string;
```

Result:

``` text
┌─h3_string───────┐
│ 89184926cdbffff │
└─────────────────┘
```

## stringToH3 {#stringtoh3}

Converts the string representation to the `H3Index` (UInt64) representation.

**Syntax**

``` sql
stringToH3(index_str)
```

**Parameter**

-   `index_str` — String representation of the H3 index. Type: [String](../../../sql-reference/data-types/string.md).

**Returned value**

-   Hexagon index number. Returns 0 on error. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT stringToH3('89184926cc3ffff') AS index;
```

Result:

``` text
┌──────────────index─┐
│ 617420388351344639 │
└────────────────────┘
```

## h3GetResolution {#h3getresolution}

Returns the resolution of the [H3](#h3index) index.

**Syntax**

``` sql
h3GetResolution(index)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-   Index resolution. Range: `[0, 15]`. Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT h3GetResolution(617420388352917503) AS res;
```

Result:

``` text
┌─res─┐
│   9 │
└─────┘
```

## h3IsResClassIII {#h3isresclassIII}

Returns whether [H3](#h3index) index has a resolution with Class III orientation.

**Syntax**

``` sql
h3IsResClassIII(index)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-   `1` — Index has a resolution with Class III orientation.
-   `0` — Index doesn't have a resolution with Class III orientation.

Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT h3IsResClassIII(617420388352917503) AS res;
```

Result:

``` text
┌─res─┐
│   1 │
└─────┘
```

## h3IsPentagon {#h3ispentagon}

Returns whether this [H3](#h3index) index represents a pentagonal cell.

**Syntax**

``` sql
h3IsPentagon(index)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-   `1` — Index represents a pentagonal cell.
-   `0` — Index doesn't represent a pentagonal cell.

Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT h3IsPentagon(644721767722457330) AS pentagon;
```

Result:

``` text
┌─pentagon─┐
│        0 │
└──────────┘
```

## h3GetFaces {#h3getfaces}

Returns icosahedron faces intersected by a given [H3](#h3index) index.

**Syntax**

``` sql
h3GetFaces(index)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   Array containing icosahedron faces intersected by a given H3 index.

Type: [Array](../../../sql-reference/data-types/array.md)([UInt64](../../../sql-reference/data-types/int-uint.md)).

**Example**

Query:

``` sql
SELECT h3GetFaces(599686042433355775) AS faces;
```

Result:

``` text
┌─faces─┐
│ [7]   │
└───────┘
```

## h3CellAreaM2 {#h3cellaream2}

Returns the exact area of a specific cell in square meters corresponding to the given input H3 index.

**Syntax**

``` sql
h3CellAreaM2(index)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-   Cell area in square meters.

Type: [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT h3CellAreaM2(579205133326352383) AS area;
```

Result:

``` text
┌───────────────area─┐
│ 4106166334463.9233 │
└────────────────────┘
```

## h3CellAreaRads2 {#h3cellarearads2}

Returns the exact area of a specific cell in square radians corresponding to the given input H3 index.

**Syntax**

``` sql
h3CellAreaRads2(index)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-   Cell area in square radians.

Type: [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT h3CellAreaRads2(579205133326352383) AS area;
```

Result:

``` text
┌────────────────area─┐
│ 0.10116268528089567 │
└─────────────────────┘
```

## h3ToCenterChild {#h3tocenterchild}

Returns the center child (finer) [H3](#h3index) index contained by given [H3](#h3index) at the given resolution.

**Syntax**

``` sql
h3ToCenterChild(index, resolution)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `resolution` — Index resolution. Range: `[0, 15]`. Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Returned values**

- [H3](#h3index) index of the center child contained by given [H3](#h3index) at the given resolution.

Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT h3ToCenterChild(577023702256844799,1) AS centerToChild;
```

Result:

``` text
┌──────centerToChild─┐
│ 581496515558637567 │
└────────────────────┘
```

## h3ExactEdgeLengthM {#h3exactedgelengthm}

Returns the exact edge length of the unidirectional edge represented by the input h3 index in meters.

**Syntax**

``` sql
h3ExactEdgeLengthM(index)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-  Exact edge length in meters.

Type: [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT h3ExactEdgeLengthM(1310277011704381439) AS exactEdgeLengthM;;
```

Result:

``` text
┌───exactEdgeLengthM─┐
│ 195449.63163407316 │
└────────────────────┘
```

## h3ExactEdgeLengthKm {#h3exactedgelengthkm}

Returns the exact edge length of the unidirectional edge represented by the input h3 index in kilometers.

**Syntax**

``` sql
h3ExactEdgeLengthKm(index)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-  Exact edge length in kilometers.

Type: [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT h3ExactEdgeLengthKm(1310277011704381439) AS exactEdgeLengthKm;;
```

Result:

``` text
┌──exactEdgeLengthKm─┐
│ 195.44963163407317 │
└────────────────────┘
```

## h3ExactEdgeLengthRads {#h3exactedgelengthrads}

Returns the exact edge length of the unidirectional edge represented by the input h3 index in radians.

**Syntax**

``` sql
h3ExactEdgeLengthRads(index)
```

**Parameter**

-   `index` — Hexagon index number. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-  Exact edge length in radians.

Type: [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT h3ExactEdgeLengthRads(1310277011704381439) AS exactEdgeLengthRads;;
```

Result:

``` text
┌──exactEdgeLengthRads─┐
│ 0.030677980118976447 │
└──────────────────────┘
```

## h3NumHexagons {#h3numhexagons}

Returns the number of unique H3 indices at the given resolution.

**Syntax**

``` sql
h3NumHexagons(resolution)
```

**Parameter**

-   `resolution` — Index resolution. Range: `[0, 15]`. Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Returned value**

-  Number of H3 indices.

Type: [Int64](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT h3NumHexagons(3) AS numHexagons;
```

Result:

``` text
┌─numHexagons─┐
│       41162 │
└─────────────┘
```
[Original article](https://clickhouse.com/docs/en/sql-reference/functions/geo/h3) <!--hide-->
