--- 
toc_title: S2 Geometry
---

# Functions for Working with S2 Index {#s2index}

[S2](https://s2geometry.io/) is a geographical indexing system where all geographical data is represented on a three-dimensional sphere (similar to a globe).

In the S2 library points are represented as the S2 Index - a specific number which encodes internally a point on the surface of a unit sphere, unlike traditional (latitude, longitude) pairs. To get the S2 point index for a given point specified in the format (latitude, longitude) use the [geoToS2](#geotos2) function. Also, you can use the [s2ToGeo](#s2togeo) function for getting geographical coordinates corresponding to the specified S2 point index.

## geoToS2 {#geotos2}

Returns [S2](#s2index) point index corresponding to the provided coordinates `(longitude, latitude)`.

**Syntax**

``` sql
geoToS2(lon, lat)
```

**Arguments**

-   `lon` — Longitude. [Float64](../../../sql-reference/data-types/float.md).
-   `lat` — Latitude. [Float64](../../../sql-reference/data-types/float.md).

**Returned values**

-   S2 point index.

Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT geoToS2(37.79506683, 55.71290588) AS s2Index;
```

Result:

``` text
┌─────────────s2Index─┐
│ 4704772434919038107 │
└─────────────────────┘
```

## s2ToGeo {#s2togeo}

Returns geo coordinates `(longitude, latitude)` corresponding to the provided [S2](#s2index) point index.

**Syntax**

``` sql
s2ToGeo(s2index)
```

**Arguments**

-   `s2index` — S2 Index. [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   A tuple consisting of two values: `tuple(lon,lat)`.

Type: `lon` — [Float64](../../../sql-reference/data-types/float.md). `lat` — [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT s2ToGeo(4704772434919038107) AS s2Coodrinates;
```

Result:

``` text
┌─s2Coodrinates────────────────────────┐
│ (37.79506681471008,55.7129059052841) │
└──────────────────────────────────────┘
```

## s2GetNeighbors {#s2getneighbors}

Returns S2 neighbor indixes corresponding to the provided [S2](#s2index). Each cell in the S2 system is a quadrilateral bounded by four geodesics. So, each cell has 4 neighbors. 

**Syntax**

``` sql
s2GetNeighbors(s2index)
```

**Arguments**

-   `s2index` — S2 Index. [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   An array consisting of 4 neighbor indexes: `array[s2index1, s2index3, s2index2, s2index4]`.

Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT s2GetNeighbors(5074766849661468672) AS s2Neighbors;
```

Result:

``` text
┌─s2Neighbors───────────────────────────────────────────────────────────────────────┐
│ [5074766987100422144,5074766712222515200,5074767536856236032,5074767261978329088] │
└───────────────────────────────────────────────────────────────────────────────────┘
```

## s2CellsIntersect {#s2cellsintersect}

Determines if the two provided [S2](#s2index) cells intersect or not.

**Syntax**

``` sql
s2CellsIntersect(s2index1, s2index2)
```

**Arguments**

-   `siIndex1`, `s2index2` — S2 Index. [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   1 — If the cells intersect.
-   0 — If the cells don't intersect.

Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT s2CellsIntersect(9926595209846587392, 9926594385212866560) AS intersect;
```

Result:

``` text
┌─intersect─┐
│         1 │
└───────────┘
```

## s2CapContains {#s2capcontains}

Determines if a cap contains a S2 point. A cap represents a part of the sphere that has been cut off by a plane. It is defined by a point on a sphere and a radius in degrees.

**Syntax**

``` sql
s2CapContains(center, degrees, point)
```

**Arguments**

-   `center` — S2 point index corresponding to the cap. [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `degrees` — Radius of the cap in degrees. [Float64](../../../sql-reference/data-types/float.md).
-   `point` — S2 point index. [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   1 — If the cap contains the S2 point index.
-   0 — If the cap doesn't contain the S2 point index.

Type: [UInt8](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT s2CapContains(1157339245694594829, 1.0, 1157347770437378819) AS capContains;
```

Result:

``` text
┌─capContains─┐
│           1 │
└─────────────┘
```

## s2CapUnion {#s2capunion}

Determines the smallest cap that contains the given two input caps. A cap represents a portion of the sphere that has been cut off by a plane. It is defined by a point on a sphere and a radius in degrees.

**Syntax**

``` sql
s2CapUnion(center1, radius1, center2, radius2)
```

**Arguments**

-   `center1`, `center2` — S2 point indixes corresponding to the two input caps. [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `radius1`, `radius2` — Radius of the two input caps in degrees. [Float64](../../../sql-reference/data-types/float.md).

**Returned values**

-   `center` — S2 point index corresponding the center of the smallest cap containing the two input caps. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `radius` — Radius of the smallest cap containing the two input caps. Type: [Float64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT s2CapUnion(3814912406305146967, 1.0, 1157347770437378819, 1.0) AS capUnion;
```

Result:

``` text
┌─capUnion───────────────────────────────┐
│ (4534655147792050737,60.2088283994957) │
└────────────────────────────────────────┘
```

## s2RectAdd {#s2rectadd}

Increases the size of the bounding rectangle to include the given S2 point. In the S2 system, a rectangle is represented by a type of S2Region called a `S2LatLngRect` that represents a rectangle in latitude-longitude space.

**Syntax**

``` sql
s2RectAdd(s2pointLow, s2pointHigh, s2Point)
```

**Arguments**

-   `s2PointLow` — Low S2 point index corresponding to the rectangle. [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `s2PointHigh` — High S2 point index corresponding to the rectangle. [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `s2Point` — Target S2 point index that the bound rectangle should be grown to include. [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   `s2PointLow` — Low S2 cell id corresponding to the grown rectangle. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `s2PointHigh` — Hight S2 cell id corresponding to the grown rectangle. Type: [UInt64](../../../sql-reference/data-types/float.md).

**Example**

Query:

``` sql
SELECT s2RectAdd(5178914411069187297, 5177056748191934217, 5179056748191934217) AS rectAdd;
```

Result:

``` text
┌─rectAdd───────────────────────────────────┐
│ (5179062030687166815,5177056748191934217) │
└───────────────────────────────────────────┘
```

## s2RectContains {#s2rectcontains}

Determines if a given rectangle contains a S2 point. In the S2 system, a rectangle is represented by a type of S2Region called a `S2LatLngRect` that represents a rectangle in latitude-longitude space.

**Syntax**

``` sql
s2RectContains(s2PointLow, s2PointHi, s2Point)
```

**Arguments**

-   `s2PointLow` — Low S2 point index corresponding to the rectangle. [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `s2PointHigh` — High S2 point index corresponding to the rectangle. [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `s2Point` — Target S2 point index. [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   1 — If the rectangle contains the given S2 point.
-   0 — If the rectangle doesn't contain the given S2 point.

**Example**

Query:

``` sql
SELECT s2RectContains(5179062030687166815, 5177056748191934217, 5177914411069187297) AS rectContains;
```

Result:

``` text
┌─rectContains─┐
│            0 │
└──────────────┘
```

## s2RectUinion {#s2rectunion}

Returns the smallest rectangle containing the union of this rectangle and the given rectangle. In the S2 system, a rectangle is represented by a type of S2Region called a `S2LatLngRect` that represents a rectangle in latitude-longitude space.

**Syntax**

``` sql
s2RectUnion(s2Rect1PointLow, s2Rect1PointHi, s2Rect2PointLow, s2Rect2PointHi)
```

**Arguments**

-   `s2Rect1PointLow`, `s2Rect1PointHi` — Low and High S2 point indexes corresponding to the first rectangle. [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `s2Rect2PointLow`, `s2Rect2PointHi` — Low and High S2 point indexes corresponding to the second rectangle. [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   `s2UnionRect2PointLow` — Low S2 cell id corresponding to the union rectangle. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `s2UnionRect2PointHi` — High S2 cell id corresponding to the union rectangle. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT s2RectUnion(5178914411069187297, 5177056748191934217, 5179062030687166815, 5177056748191934217) AS rectUnion;
```

Result:

``` text
┌─rectUnion─────────────────────────────────┐
│ (5179062030687166815,5177056748191934217) │
└───────────────────────────────────────────┘
```

## s2RectIntersection {#s2rectintersection}

Returns the smallest rectangle containing the intersection of this rectangle and the given rectangle. In the S2 system, a rectangle is represented by a type of S2Region called a `S2LatLngRect` that represents a rectangle in latitude-longitude space.

**Syntax**

``` sql
s2RectIntersection(s2Rect1PointLow, s2Rect1PointHi, s2Rect2PointLow, s2Rect2PointHi)
```

**Arguments**

-   `s2Rect1PointLow`, `s2Rect1PointHi` — Low and High S2 point indexes corresponding to the first rectangle. [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `s2Rect2PointLow`, `s2Rect2PointHi` — Low and High S2 point indexes corresponding to the second rectangle. [UInt64](../../../sql-reference/data-types/int-uint.md).

**Returned values**

-   `s2UnionRect2PointLow` — Low S2 cell id corresponding to the rectangle containing the intersection of the given rectangles. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).
-   `s2UnionRect2PointHi` — High S2 cell id corresponding to the rectangle containing the intersection of the given rectangles. Type: [UInt64](../../../sql-reference/data-types/int-uint.md).

**Example**

Query:

``` sql
SELECT s2RectIntersection(5178914411069187297, 5177056748191934217, 5179062030687166815, 5177056748191934217) AS rectIntersection;
```

Result:

``` text
┌─rectIntersection──────────────────────────┐
│ (5178914411069187297,5177056748191934217) │
└───────────────────────────────────────────┘
```
